<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Campaign;
use App\Models\Transaction;
use Illuminate\Support\Facades\DB;

class CampaignController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = auth()->user();
        if ($user->isBrand()) {
            $campaigns = $user->campaigns()->latest()->get();
            return view('campaigns.index', compact('campaigns'));
        } else {
            $campaigns = Campaign::where('status', 'active')->latest()->get();
            return view('campaigns.browse', compact('campaigns'));
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('campaigns.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'whatsapp_link' => 'required|url',
            'rate_per_member' => 'required|numeric|min:0.01',
            'target_users' => 'required|integer|min:1',
        ]);

        $user = auth()->user();

        // Ensure wallet exists
        if (!$user->wallet) {
            $user->wallet()->create(['balance' => 0, 'pending_balance' => 0]);
            $user->load('wallet');
        }

        $totalCost = $validated['rate_per_member'] * $validated['target_users'];
        $wallet = $user->wallet;

        if ($wallet->balance < $totalCost) {
            return back()->with('error', 'Insufficient wallet balance. Please fund your wallet first.')->withInput();
        }

        DB::transaction(function () use ($user, $wallet, $validated, $totalCost) {
            $campaign = $user->campaigns()->create(array_merge($validated, [
                'total_cost' => $totalCost,
                'status' => 'pending'
            ]));

            $wallet->decrement('balance', $totalCost);

            Transaction::create([
                'wallet_id' => $wallet->id,
                'amount' => $totalCost,
                'type' => 'campaign_budget',
                'status' => 'completed',
                'description' => "Budget deduction for campaign: {$campaign->title}",
            ]);
        });

        return redirect()->route('campaigns.index')->with('success', 'Campaign created successfully and cost deducted from your wallet.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Campaign $campaign)
    {
        $user = auth()->user();

        // Admin can see everything
        if ($user->isAdmin()) {
            return view('campaigns.show', compact('campaign'));
        }

        // Brand can see their own campaigns
        if ($user->isBrand() && $campaign->brand_id === $user->id) {
            return view('campaigns.show', compact('campaign'));
        }

        // Promoters can only see active campaigns
        if ($user->isPromoter() && $campaign->status === 'active') {
            return view('campaigns.show', compact('campaign'));
        }

        abort(403, 'Unauthorized action.');
    }

    /**
     * Promoter joins a campaign.
     */
    public function join(Campaign $campaign)
    {
        $user = auth()->user();
        if (!$user->isPromoter()) {
            abort(403);
        }

        if ($campaign->promoters()->where('promoter_id', $user->id)->exists()) {
            return back()->with('info', 'You have already joined this campaign.');
        }

        $campaign->promoters()->attach($user->id, ['status' => 'joined']);

        return back()->with('success', 'You have joined the campaign! Follow the instructions to start earning.');
    }
}
