<?php

namespace App\Http\Controllers;

use App\Models\ProofSubmission;
use App\Models\Transaction;
use App\Models\Campaign;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{
    public function submissions()
    {
        $submissions = ProofSubmission::with(['campaign', 'promoter'])->latest()->paginate(20);
        return view('admin.submissions.index', compact('submissions'));
    }

    public function approveSubmission(Request $request, ProofSubmission $submission)
    {
        if ($submission->status !== 'pending') {
            return back()->with('error', 'This submission has already been processed.');
        }

        DB::transaction(function () use ($submission) {
            $submission->update(['status' => 'approved']);

            $promoter = $submission->promoter;
            $wallet = $promoter->wallet;
            $campaign = $submission->campaign;

            // Assuming 1 member growth for now as per simple MVP or should be manual?
            // The prompt says "Promoters submit proof of member growth"
            // Let's assume the admin enters the growth count or we use a fixed amount.
            $amount = $campaign->rate_per_member; // Basic logic: 1 approved proof = 1 rate unit? 
            // Or maybe the admin enters the count. Let's add that to the request.

            $growth = request('member_growth', 1);
            $totalEarning = $growth * $amount;

            $wallet->increment('balance', $totalEarning);

            Transaction::create([
                'wallet_id' => $wallet->id,
                'amount' => $totalEarning,
                'type' => 'earning',
                'status' => 'completed',
                'description' => "Earning from campaign: {$campaign->title}",
            ]);

            // Up reputation
            $promoter->increment('trust_score', 2);
        });

        return back()->with('success', 'Submission approved and promoter paid.');
    }

    public function rejectSubmission(Request $request, ProofSubmission $submission)
    {
        $submission->update([
            'status' => 'rejected',
            'admin_comment' => $request->comment
        ]);

        // Down reputation
        $submission->promoter->decrement('trust_score', 5);

        return back()->with('success', 'Submission rejected.');
    }

    public function campaigns()
    {
        $campaigns = Campaign::with('brand')->latest()->paginate(20);
        return view('admin.campaigns.index', compact('campaigns'));
    }

    public function approveCampaign(Campaign $campaign)
    {
        $campaign->update(['status' => 'active']);
        return back()->with('success', 'Campaign approved and is now active.');
    }

    public function rejectCampaign(Campaign $campaign)
    {
        $campaign->update(['status' => 'paused']);
        return back()->with('success', 'Campaign has been paused/rejected.');
    }

    public function updateCampaignLink(Request $request, Campaign $campaign)
    {
        $request->validate([
            'whatsapp_link' => 'required|url',
        ]);

        $campaign->update([
            'whatsapp_link' => $request->whatsapp_link
        ]);

        return back()->with('success', 'Campaign link updated successfully.');
    }

    public function users()
    {
        $users = User::with('wallet')->latest()->paginate(20);
        return view('admin.users.index', compact('users'));
    }

    public function fundWallet(Request $request, User $user)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'description' => 'nullable|string|max:255',
        ]);

        DB::transaction(function () use ($user, $request) {
            if (!$user->wallet) {
                $user->wallet()->create(['balance' => 0, 'pending_balance' => 0]);
                $user->load('wallet');
            }

            $user->wallet->increment('balance', $request->amount);

            Transaction::create([
                'wallet_id' => $user->wallet->id,
                'amount' => $request->amount,
                'type' => 'deposit',
                'status' => 'completed',
                'description' => $request->description ?? 'Manual funding by Admin',
            ]);
        });

        return back()->with('success', "Wallet for {$user->name} funded successfully.");
    }

    public function dashboard()
    {
        $stats = [
            'pending_submissions' => ProofSubmission::where('status', 'pending')->count(),
            'total_campaigns' => \App\Models\Campaign::count(),
            'total_users' => \App\Models\User::count(),
        ];
        return view('admin.dashboard', compact('stats'));
    }

    public function manageLinks(Campaign $campaign)
    {
        $campaign->load('links');
        return view('admin.campaigns.links', compact('campaign'));
    }

    public function storeLinks(Request $request, Campaign $campaign)
    {
        $request->validate([
            'links' => 'required|string',
        ]);

        $links = explode("\n", str_replace("\r", "", $request->links));
        $count = 0;

        foreach ($links as $linkText) {
            $linkText = trim($linkText);
            if (filter_var($linkText, FILTER_VALIDATE_URL)) {
                $campaign->links()->create([
                    'group_link' => $linkText,
                    'is_active' => true
                ]);
                $count++;
            }
        }

        return back()->with('success', "{$count} WhatsApp links added successfully.");
    }

    public function deleteLink(\App\Models\CampaignLink $link)
    {
        $link->delete();
        return back()->with('success', 'WhatsApp link removed.');
    }
}
